#!/bin/bash -l

# EBSスナップショットを採取する

# EBSボリュームID
EBS_VOLUME_ID=vol-xxxx

# スナップショットを採取するEBSが存在しているリージョン
REGION_FROM=xxxx
# スナップショットのコピー先とするリージョン
REGION_TO=xxxx

# ノードを識別するための名前
NODE_NAME=NODExx
# adminユーザ名
ADMIN_USER=admin
# adminパスワード
ADMIN_PASS=admin

# バックアップを保持する世代数
HOLD_GEN_CNT=3

# 世代番号管理ファイルのパス
DR_BACKUP_GEN_FILE_PATH=${GS_HOME}/dr_backup
DR_BACKUP_GEN_FILE_NAME=dr_backup_gen.txt
DR_BACKUP_GEN_FILE=${DR_BACKUP_GEN_FILE_PATH}/${DR_BACKUP_GEN_FILE_NAME}
DR_BACKUP_GEN_CPFROM_FILE_NAME=dr_backup_cp_from_gen.txt
DR_BACKUP_GEN_CPFROM_FILE=${DR_BACKUP_GEN_FILE_PATH}/${DR_BACKUP_GEN_CPFROM_FILE_NAME}

# ログ出力先のパス
LOG_PATH=${GS_HOME}/dr_backup
# ログファイル名
LOG_NAME=dr_backup_${NODE_NAME}.log

function log () {
  # ログ出力先
  LOG=${LOG_PATH}/${LOG_NAME}
  # ログ出力日時
  time=$(date +"%Y-%m-%d %H:%M:%S")

  # 引数の文字列をログ出力する
  echo -e "${time} $1 $2" >> $LOG
}

function err_exit () {
  exit 1
}


# 定期チェックポイント処理を停止
gs_checkpoint -u ${ADMIN_USER}/${ADMIN_PASS} --off

# 手動チェックポイントを行う
gs_checkpoint -u admin/admin --manual -w


log "INFO" "START create snapshot ${EBS_VOLUME_ID}"

# EBSスナップショットを作成
CREATE_SNAPSHOT_R=$(aws ec2 create-snapshot --volume-id ${EBS_VOLUME_ID} --description "DR backup ${NODE_NAME}" --region ${REGION_FROM})
if [ $? -ne 0 ]; then
  log "ERROR" "failed to create-snapshot"
  # 定期チェックポイント処理を再開
  gs_checkpoint -u ${ADMIN_USER}/${ADMIN_PASS} --on
  err_exit
fi

log "INFO" "END create snapshot ${EBS_VOLUME_ID}"

# 定期チェックポイント処理を再開
gs_checkpoint -u ${ADMIN_USER}/${ADMIN_PASS} --on

# スナップショットID取得
SNAPID=$(echo "${CREATE_SNAPSHOT_R}" | grep SnapshotId)
SNAPID=$(echo ${SNAPID} | sed -e "s/^[\t]*\"SnapshotId\"://")
SNAPID=$(echo ${SNAPID} | sed -e "s/\"[\t]*$//")
SNAPID=$(echo ${SNAPID} | sed -e "s/^\"//")

log "INFO" "snapshotId:${SNAPID}"

# スナップショット作成完了を待つ

log "INFO" "START wait for complete snapshot ${SNAPID} "

WAIT_TIMEOUT=1
for i in $(seq 1440)
do
  sleep 60
  
  SNAPDESC=$(aws ec2 describe-snapshots  --region ${REGION_FROM} --filters Name=snapshot-id,Values=${SNAPID})
  SNAP_STATUS=$(echo "${SNAPDESC}" | grep \"State\")
  SNAP_PROGRESS=$(echo "${SNAPDESC}" | grep \"Progress\")
  TMP=$(echo $SNAP_STATUS | grep completed)
  if [ $? -eq 0 ]; then
    WAIT_TIMEOUT=0
    break
  fi
  
  log "INFO" "wait for complete snapshot ${SNAPID} ${SNAP_PROGRESS}"
done

if [ ${WAIT_TIMEOUT} -eq 1 ]; then
  log ERROR "wait for complete snapshot timeout"
  err_exit
fi

log "INFO" "END wait for complete snapshot ${SNAPID} "

# 世代管理ファイル 更新
if [ ! -d ${DR_BACKUP_GEN_FILE_PATH} ]; then
  mkdir -p ${DR_BACKUP_GEN_FILE_PATH}
fi
echo ${SNAPID} >> ${DR_BACKUP_GEN_CPFROM_FILE}

CPFROM_SNAPID_LIST=()
OUTPUT_CPFROM_SNAPID_LIST=()
# 過去のスナップショットを削除する
if [ -f ${DR_BACKUP_GEN_CPFROM_FILE} ]; then
  while read line
  do
    if [ "${line}" != "" ]; then
      CPFROM_SNAPID_LIST+=(${line})
    fi
  done < ${DR_BACKUP_GEN_CPFROM_FILE}
fi

# 過去のスナップショットを削除
i=0
if [ ${#CPFROM_SNAPID_LIST[@]} -ge 2 ]; then
  while [ ${i} -lt ${#CPFROM_SNAPID_LIST[@]} ]
  do
    if [ ${i} -ge $((${#CPFROM_SNAPID_LIST[@]} -1)) ]; then
      OUTPUT_CPFROM_SNAPID_LIST+=(${CPFROM_SNAPID_LIST[${i}]})
    else
      DELETE_SNAP_ID=${CPFROM_SNAPID_LIST[${i}]}
      log "INFO" "START delete snapshot ${DELETE_SNAP_ID} ${REGION_FROM}"
      aws ec2 delete-snapshot --snapshot-id ${DELETE_SNAP_ID} --region ${REGION_FROM}
      if [ $? -ne 0 ]; then
        OUTPUT_CPFROM_SNAPID_LIST+=(${DELETE_SNAP_ID})
        log "ERROR" "failed to delete-snapshot ${DELETE_SNAP_ID} ${REGION_FROM}"
      else
        log "INFO" "END delete snapshot ${DELETE_SNAP_ID} ${REGION_FROM}"
      fi
    fi
  
    i=`expr ${i} + 1`
  done
else
  OUTPUT_CPFROM_SNAPID_LIST=${CPFROM_SNAPID_LIST[@]}
fi

# 世代管理ファイル 更新
echo -n > ${DR_BACKUP_GEN_CPFROM_FILE}
for OUTPUT_SNAPID in "${OUTPUT_CPFROM_SNAPID_LIST[@]}"
do
    echo ${OUTPUT_SNAPID} >> ${DR_BACKUP_GEN_CPFROM_FILE}
done


# EBSスナップショットをコピー
COPY_SNAPSHOT_R=$(aws --region ${REGION_TO} ec2 copy-snapshot --description "DR backup ${NODE_NAME}" --source-region ${REGION_FROM} --source-snapshot-id ${SNAPID})
if [ $? -ne 0 ]; then
  log "ERROR" "failed to copy-snapshot"
  err_exit
fi

CP_SNAPID=$(echo "${COPY_SNAPSHOT_R}" | grep SnapshotId)
CP_SNAPID=$(echo ${CP_SNAPID} | sed -e "s/^[\t]*\"SnapshotId\"://")
CP_SNAPID=$(echo ${CP_SNAPID} | sed -e "s/\"[\t]*$//")
CP_SNAPID=$(echo ${CP_SNAPID} | sed -e "s/^\"//")

log "INFO" "copy snapshot snapshotId:${SNAPID} ${REGION_FROM} to snapshotId:${CP_SNAPID} ${REGION_TO}"


# スナップショットコピー完了を待つ

log "INFO" "START wait for complete copy snapshot ${CP_SNAPID} "

WAIT_TIMEOUT=1
for i in $(seq 1440)
do
  sleep 60
  
  SNAPDESC=$(aws ec2 describe-snapshots  --region ${REGION_TO} --filters Name=snapshot-id,Values=${CP_SNAPID})
  SNAP_STATUS=$(echo "${SNAPDESC}" | grep \"State\")
  SNAP_PROGRESS=$(echo "${SNAPDESC}" | grep \"Progress\")
  TMP=$(echo $SNAP_STATUS | grep completed)
  if [ $? -eq 0 ]; then
    WAIT_TIMEOUT=0
    break
  fi
  
  log "INFO" "wait for complete snapshot ${CP_SNAPID} ${SNAP_PROGRESS}"
done

if [ ${WAIT_TIMEOUT} -eq 1 ]; then
  log ERROR "wait for complete snapshot timeout"
  err_exit
fi

log "INFO" "END wait for complete copy snapshot ${CP_SNAPID} "


# 世代管理ファイル 更新
echo ${CP_SNAPID} >> ${DR_BACKUP_GEN_FILE}

CPTO_SNAPID_LIST=()
OUTPUT_CPTO_SNAPID_LIST=()
# 過去のスナップショットを削除する
if [ -f ${DR_BACKUP_GEN_FILE} ]; then
  while read line
  do
    if [ "${line}" != "" ]; then
      CPTO_SNAPID_LIST+=(${line})
    fi
  done < ${DR_BACKUP_GEN_FILE}
fi

# 過去のスナップショットを削除
if [ ${#CPTO_SNAPID_LIST[@]} -ge ${HOLD_GEN_CNT} ]; then
  i=0
  while [ ${i} -lt ${#CPTO_SNAPID_LIST[@]} ]
  do
    if [ ${i} -ge $((${#CPTO_SNAPID_LIST[@]} - ${HOLD_GEN_CNT})) ]; then
      OUTPUT_CPTO_SNAPID_LIST+=(${CPTO_SNAPID_LIST[${i}]})
    else
      DELETE_SNAP_ID=${CPTO_SNAPID_LIST[${i}]}
      log "INFO" "START delete snapshot ${DELETE_SNAP_ID} ${REGION_TO}"
      aws ec2 delete-snapshot --snapshot-id ${DELETE_SNAP_ID} --region ${REGION_TO}
      if [ $? -ne 0 ]; then
        OUTPUT_CPTO_SNAPID_LIST+=(${DELETE_SNAP_ID})
        log "ERROR" "failed to delete-snapshot ${DELETE_SNAP_ID} ${REGION_TO}"
      else
        log "INFO" "END delete snapshot ${DELETE_SNAP_ID} ${REGION_TO}"
      fi
    fi

    i=`expr ${i} + 1`
  done
else
  OUTPUT_CPTO_SNAPID_LIST=${CPTO_SNAPID_LIST[@]}
fi

# 世代管理ファイル 更新
echo -n > ${DR_BACKUP_GEN_FILE}
for OUTPUT_SNAPID in "${OUTPUT_CPTO_SNAPID_LIST[@]}"
do
    echo ${OUTPUT_SNAPID} >> ${DR_BACKUP_GEN_FILE}
done

exit 0


