#!/bin/bash -l

# トランザクションログファイルをコピーした後に、データファイルとチェックポイントログファイルをコピー

# エンドポイントDNS名
ENDPOINT_DNS=xxxx.xxxx.vpce.amazonaws.com

# S3 バケット名
S3BUCKET_NAME=xxxx
# バックアップ先リージョン
BACKUP_REGION=xxxx

# ノードを識別するための名前
NODE_NAME=NODExx
# adminユーザ名
ADMIN_USER=admin
# adminパスワード
ADMIN_PASS=admin
# txnlogディレクトリのパス
TXNLOG_PATH=${GS_HOME}/txnlog
# dataディレクトリのパス
DATA_PATH=${GS_HOME}/data

# 自動ログバックアップ作成ディレクトリ
LOCAL_BACKUP_PATH=${GS_HOME}/backup

# バックアップを保持する世代数
HOLD_GEN_CNT=3

# 世代番号管理ファイルのパス
DR_BACKUP_GEN_FILE_PATH=${GS_HOME}/dr_backup
DR_BACKUP_GEN_FILE_NAME=dr_backup_gen.txt
DR_BACKUP_GEN_FILE=${DR_BACKUP_GEN_FILE_PATH}/${DR_BACKUP_GEN_FILE_NAME}
# S3上のバックアップのプレフィックスとして使用する名前
DR_BACKUP_PREFIX=dr_backup

# ログ出力先のパス
LOG_PATH=${GS_HOME}/dr_backup
# ログファイル名
LOG_NAME=dr_backup_${NODE_NAME}.log

# 自動ログバックアップのバックアップ名
AUTO_BACKUP_NAME=drautobkup
# ダミーとして使用する自動ログバックアップのバックアップ名
AUTO_BACKUP_DUMMY_NAME=drautobkupd

# ロックファイル
LOCK_PATH=${GS_HOME}/dr_backup
LOCK_FILE_FULLBAKUP_NAME=.lock_fullbk
LOCK_FILE_FULLBAKUP=${LOCK_PATH}/${LOCK_FILE_FULLBAKUP_NAME}
LOCK_FILE_AUTOBAKUP_NAME=.lock_autobk
LOCK_FILE_AUTOBAKUP=${LOCK_PATH}/${LOCK_FILE_AUTOBAKUP_NAME}
PS_NAME_FULL=full_bkup
PS_NAME_AUTO=auto_bkup

ENDPOINT_URL=https://bucket.${ENDPOINT_DNS}


function log () {
  # ログ出力先
  LOG=${LOG_PATH}/${LOG_NAME}
  # ログ出力日時
  time=$(date +"%Y-%m-%d %H:%M:%S")

  # 引数の文字列をログ出力する
  echo -e "${time} $1 $2" >> $LOG
}

function err_exit () {
  # ロックファイル削除
  unlink ${LOCK_FILE_FULLBAKUP}
  exit 1
}

function check_lock_file () {
  # ロックファイルの有無チェック
  if [ -f ${LOCK_FILE_FULLBAKUP} ]; then
    FULL_PID=$(cat ${LOCK_FILE_FULLBAKUP})
    PS=$(ps aux | grep ${FULL_PID} | grep ${PS_NAME_FULL} | grep -v grep | wc -l)
    if [ ${PS} -eq 0 ]; then
      log INFO "${LOCK_FILE_FULLBAKUP} exists. But process do not exists."
    else
      # ロックファイルが存在し、そのプロセスが残っている場合、バックアップの作成をスキップ
      log ERROR "${LOCK_FILE_FULLBAKUP} exists. skip backup."
      err_exit
    fi
  fi
}

function check_auto_lock_file () {
  DO_SKIP=1
  for i in $(seq 10)
  do
    # 自動ログバックアップのコピーを行うプロセスのロックファイルの有無チェック
    if [ -f ${LOCK_FILE_AUTOBAKUP} ]; then
      AUTO_PID=$(cat ${LOCK_FILE_AUTOBAKUP})
      PS=$(ps aux | grep ${AUTO_PID} | grep ${PS_NAME_AUTO} | grep -v grep | wc -l)
      if [ ${PS} -eq 0 ]; then
        log INFO "${LOCK_FILE_AUTOBAKUP} exists. But process do not exists."
        DO_SKIP=0
        break
      else
        sleep 60
        log INFO "${LOCK_FILE_AUTOBAKUP} exists. wait for completion of copy auto backup."
      fi
    else
      log INFO "${LOCK_FILE_AUTOBAKUP} not exists."
      DO_SKIP=0
      break
    fi
  done
  
  if [ ${DO_SKIP} -eq 1 ]; then
    log ERROR "${LOCK_FILE_AUTOBAKUP} exists. skip backup."
    err_exit
  fi
}


# ロックファイルの有無チェック
check_lock_file
# ロックファイル作成
echo $$ > ${LOCK_FILE_FULLBAKUP}
# 自動ログバックアップコピーのロックファイルの有無チェック
check_auto_lock_file

# 作成するバックアップの世代番号を求める
if [ -f ${DR_BACKUP_GEN_FILE} ]; then
  BACKUP_GEN_NUM=$(cat ${DR_BACKUP_GEN_FILE})
else
  BACKUP_GEN_NUM=0
fi
BACKUP_GEN_NUM=$(expr ${BACKUP_GEN_NUM} + 1)

BACKUP_GEN_PATH_NAME=${DR_BACKUP_PREFIX}${BACKUP_GEN_NUM}


# 定期チェックポイント処理を停止
gs_checkpoint -u ${ADMIN_USER}/${ADMIN_PASS} --off

log "INFO" "START auto backup ${AUTO_BACKUP_NAME}"

# 自動ログバックアップが存在しているかチェック
DEL_AUTO_BACKUP_NAME=0
if [ -d ${LOCAL_BACKUP_PATH}/${AUTO_BACKUP_NAME} ]; then
  DEL_AUTO_BACKUP_NAME=1
fi

# 既に自動ログバックアップが存在している場合、削除する
if [ ${DEL_AUTO_BACKUP_NAME} -eq 1 ]; then
  # 現在出力中の自動ログバクアップを停止させるために、
  # ダミーの自動ログバックアップ作成を行う
  rm -rf ${LOCAL_BACKUP_PATH}/${AUTO_BACKUP_DUMMY_NAME}
  gs_backup -u admin/admin --mode auto --skipBaseline ${AUTO_BACKUP_DUMMY_NAME}
  sleep 10
  # 既存の自動ログバックアップを削除する
  rm -rf ${LOCAL_BACKUP_PATH}/${AUTO_BACKUP_NAME}
fi

# 自動ログバックアップを開始する
gs_backup -u admin/admin --mode auto --skipBaseline ${AUTO_BACKUP_NAME}
if [ $? -ne 0 ]; then
  log "ERROR" "failed auto backup"
  err_exit
fi
sleep 10

# 手動チェックポイントを行う
gs_checkpoint -u admin/admin --manual -w

# ダミーの自動ログバックアップを削除する
rm -rf ${LOCAL_BACKUP_PATH}/${AUTO_BACKUP_DUMMY_NAME}

log "INFO" "END auto backup ${AUTO_BACKUP_NAME}"


log "INFO" "START backup to ${S3BUCKET_NAME}/${NODE_NAME}/${BACKUP_GEN_PATH_NAME}/fullbackup"

# トランザクションログファイルをS3へコピー
aws s3 --region ${BACKUP_REGION} --endpoint-url ${ENDPOINT_URL} cp ${TXNLOG_PATH} s3://${S3BUCKET_NAME}/${NODE_NAME}/${BACKUP_GEN_PATH_NAME}/fullbackup/txnlog/ --recursive
if [ $? -ne 0 ]; then
  log "ERROR" "failed to backup txnlog"
  err_exit
fi

# データファイルとチェックポイントログファイルをS3へコピー
aws s3 --region ${BACKUP_REGION} --endpoint-url ${ENDPOINT_URL} cp ${DATA_PATH} s3://${S3BUCKET_NAME}/${NODE_NAME}/${BACKUP_GEN_PATH_NAME}/fullbackup/data/ --recursive
if [ $? -ne 0 ]; then
  log "ERROR" "failed to backup data"
  err_exit
fi

log "INFO" "END backup to ${S3BUCKET_NAME}/${NODE_NAME}/${BACKUP_GEN_PATH_NAME}/fullbackup"

# 世代番号管理ファイル 更新
if [ ! -d ${DR_BACKUP_GEN_FILE_PATH} ]; then
  mkdir -p ${DR_BACKUP_GEN_FILE_PATH}
fi
echo ${BACKUP_GEN_NUM} > ${DR_BACKUP_GEN_FILE}

# 定期チェックポイント処理を再開
gs_checkpoint -u ${ADMIN_USER}/${ADMIN_PASS} --on



# S3上に存在するバックアップの世代番号を取得する
TMP=$(aws s3 --region ${BACKUP_REGION} --endpoint-url ${ENDPOINT_URL} ls s3://${S3BUCKET_NAME}/${NODE_NAME}/)

BACKUP_GEN_NUM_ARR=()
for S in ${TMP[@]}
do
  SSTR=${S:0:9}
  if [ ${SSTR} == ${DR_BACKUP_PREFIX} ]; then
    # S3上に存在するバックアップのプレフィックス名の一部を切り出してバックアップの世代番号を取得する
    TMP_BACKUP_GEN_NUM=${S:9:${#S} - 1}
    # バックアップの世代番号を配列に追加
    BACKUP_GEN_NUM_ARR+=(${TMP_BACKUP_GEN_NUM})
  fi
done

# バックアップの世代番号を降順ソート
BACKUP_GEN_NUM_ARR_SORT=$(printf "%s\n" "${BACKUP_GEN_NUM_ARR[@]}" | sort -n -r)

CNT=1
for CHECK_BACKUP_GEN_NUM in ${BACKUP_GEN_NUM_ARR_SORT[@]}
do
  # 最新のバックアップからバックアップを保持する世代数分のバックアップは削除しない
  # それ以降の古いバックアップは削除する
  if [ ${CNT} -gt ${HOLD_GEN_CNT} ]; then
    REMOVE_BACKUP_GEN_PATH_NAME=${DR_BACKUP_PREFIX}${CHECK_BACKUP_GEN_NUM}
    log "INFO" "START remove backup ${REMOVE_BACKUP_GEN_PATH_NAME}"
    # 古いバックアップを削除する
    aws s3 --region ${BACKUP_REGION} --endpoint-url ${ENDPOINT_URL} rm s3://${S3BUCKET_NAME}/${NODE_NAME}/${REMOVE_BACKUP_GEN_PATH_NAME} --recursive
    if [ $? -ne 0 ]; then
      log "ERROR" "failed to remove backup"
      err_exit
    fi
    log "INFO" "END remove backup ${REMOVE_BACKUP_GEN_PATH_NAME}"
  fi
  CNT=$(expr ${CNT} + 1)
done

# ロックファイル削除
unlink ${LOCK_FILE_FULLBAKUP}

exit 0


